/*+ EnumSer.cpp
 *
 ******************************************************************************
 *
 *                        Trimble Navigation Limited
 *                           645 North Mary Avenue
 *                              P.O. Box 3642
 *                         Sunnyvale, CA 94088-3642
 *
 ******************************************************************************
 *
 *    Copyright  2005 Trimble Navigation Ltd.
 *    All Rights Reserved
 *
 ******************************************************************************
 *
 * Description:
 *    This file implements the CEnumSer class - a simple function to enumerate 
 *    the serial ports installed .
 *            
 * Revision History:
 *    Created: PJN / 03-10-1998
 *    History: 
 *       PJN / 23-02-1999 
 *              Code now uses QueryDosDevice if running on NT to determine 
 *              which serial ports are available. This avoids having to open 
 *              the ports at all. It should operate a lot faster in addition.
 *       PJN / 12-12-1999 
 *              Fixed a problem in the Win9x code path when trying to detect 
 *              deactivated IRDA-ports. When trying to open those, you will 
 *              get the error-code ERROR_GEN_FAILURE.
 *       PJN / 17-05-2000 
 *              Code now uses GetDefaultCommConfig in all cases to detect 
 *              the ports.
 *       PJN / 29-03-2001 
 *              1. Reverted code to use CreateFile or QueryDosDevice as it is 
 *              much faster than using the GetDefaultCommConfig method
 *              2. Updated copyright message
 *       PJN / 25-06-2001 
 *              1. Guess what, You now have the choice of using the 
 *              GetDefaultCommConfig thro the use of three versions of the 
 *              function. You take your pick.
 *              2. Fixed problem where port fails to be reported thro the 
 *              CreateFile mechanism when the error code is 
 *              ERROR_SHARING_VIOLATION i.e. someone has the port already open
 *       PJN / 11-08-2001 
 *              1. Made code path which uses QueryDosDevice more robust by 
 *              checking to make sure the device name is of the form "COMxyz.." 
 *              where xyz are numeric
 *       PJN / 13-08-2001 
 *              1. Made the code in IsNumeric more robust when sent an empty 
 *              string
 *              2. Optimized the code in EnumerateSerialPorts2 somewhat. Thanks 
 *              to Dennis Lim for these suggestions.
 *       Mike Priven / 18-05-2005
 *              Adopted for this project.
 *
 * Notes:
 *
-*/

/*---------------------------------------------------------------------------*\
 |                         I N C L U D E   F I L E S
\*---------------------------------------------------------------------------*/
#include "stdafx.h"
#include "EnumSer.h"


/*---------------------------------------------------------------------------*\
 |                  C O N S T A N T S   A N D   M A C R O S
\*---------------------------------------------------------------------------*/
#ifdef _DEBUG
#undef THIS_FILE
#define new DEBUG_NEW
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

#define NUM_PORTS_TO_CHECK 64


/*---------------------------------------------------------------------------*\
 |                    M E T H O D   D E F I N I T I O N S
\*---------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
Function:       EnumSerialPorts

Description:    Retrieves all serial communication port numbers from the
                systems and updates the specified array.

Parameters:     arrPorts - array of available COM port numbers

Return Value:   none
-----------------------------------------------------------------------------*/
void CEnumSer::EnumSerialPorts (CUIntArray& arrPorts)
{
    CUIntArray arrPorts1, arrPorts2, arrPorts3;
    int i;

    if (!EnumMethod2(arrPorts2))
    {
        EnumMethod1(arrPorts1);
        EnumMethod3(arrPorts3);
    }

    arrPorts.RemoveAll();

    for (i=0; i<arrPorts1.GetSize(); i++)
    {
        AddData(arrPorts,arrPorts1[i]);
    }
    for (i=0; i<arrPorts2.GetSize(); i++)
    {
        AddData(arrPorts,arrPorts2[i]);
    }
    for (i=0; i<arrPorts3.GetSize(); i++)
    {
        AddData(arrPorts,arrPorts3[i]);
    }
}

/*-----------------------------------------------------------------------------
Function:       EnumMethod1

Description:    Uses the CreateFile method to retrieves com port info.

Parameters:     ports - array of available COM port numbers

Return Value:   none
-----------------------------------------------------------------------------*/
BOOL CEnumSer::EnumMethod1(CUIntArray& ports)
{
    // Make sure we clear out any elements which may already be in the array
    ports.RemoveAll();

    // Up to 255 COM ports are supported so we iterate through all of them 
    // seeing if we can open them or if we fail to open them, get an access 
    // denied or general error error. Both of these cases indicate that there 
    // is a COM port at that number. 
    for (UINT i=1; i<=NUM_PORTS_TO_CHECK; i++)
    {
        // Form the Raw device name
        CString sPort;
        sPort.Format(_T("\\\\.\\COM%d"), i);

        // Try to open the port
        BOOL bSuccess = FALSE;
        HANDLE hPort = ::CreateFile(sPort, GENERIC_READ | GENERIC_WRITE, 0, 0, 
                                    OPEN_EXISTING, 0, 0);
        if (hPort == INVALID_HANDLE_VALUE)
        {
            DWORD dwError = GetLastError();

            // Check to see if the error was because some other app had the 
            // port open or a general failure
            if (dwError == ERROR_ACCESS_DENIED || 
                dwError == ERROR_GEN_FAILURE || 
                dwError == ERROR_SHARING_VIOLATION)
            {
                bSuccess = TRUE;
            }
        }
        else
        {
            // The port was opened successfully
            bSuccess = TRUE;

            // Don't forget to close the port, since we are going to do nothing 
            // with it anyway
            CloseHandle(hPort);
        }

        // Add the port number to the array which will be returned
        if (bSuccess)
        {
            ports.Add(i);
        }
    }

    return TRUE;
}

/*-----------------------------------------------------------------------------
Function:       IsNumeric

Description:    Help function used by QueryDosDevice

Parameters:     pszString - string to check if numeric

Return Value:   none
-----------------------------------------------------------------------------*/
BOOL CEnumSer::IsNumeric(LPCTSTR pszString)
{
    int nLen = _tcslen(pszString);

    if (nLen == 0)
    {
        return FALSE;
    }

    // Assume the best
    BOOL bNumeric = TRUE;

    for (int i=0; i<nLen && bNumeric; i++)
    {
        bNumeric = (_istdigit(pszString[i]) != 0);
    }

    return bNumeric;
}

/*-----------------------------------------------------------------------------
Function:       EnumMethod2

Description:    Uses the QueryDosDevice method to retrieves com port info.

Parameters:     ports - array of available COM port numbers

Return Value:   none
-----------------------------------------------------------------------------*/
BOOL CEnumSer::EnumMethod2(CUIntArray& ports)
{
    // Make sure we clear out any elements which may already be in the array
    ports.RemoveAll();

    // Determine what OS we are running on
    OSVERSIONINFO osvi;
    osvi.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
    BOOL bGetVer = GetVersionEx(&osvi);

    // On NT use the QueryDosDevice API
    if (bGetVer && (osvi.dwPlatformId == VER_PLATFORM_WIN32_NT))
    {
        // Use QueryDosDevice to look for all devices of the form COMx. This 
        // is a better solution as it means that no ports have to be opened 
        // at all.
        TCHAR szDevices[65535];
        DWORD dwChars = QueryDosDevice(NULL, szDevices, 65535);
        if (dwChars)
        {
            int i=0;
            while (szDevices[i] != _T('\0'))
            {
                // Get the current device name
                TCHAR* pszCurrentDevice = &szDevices[i];

                // If it looks like "COMX" then
                // add it to the array which will be returned
                int nLen = _tcslen(pszCurrentDevice);
                if (nLen > 3)
                {
                    if ((_tcsnicmp(pszCurrentDevice, _T("COM"), 3) == 0) && 
                        IsNumeric(&pszCurrentDevice[3]))
                    {
                        // Work out the port number
                        int nPort = _ttoi(&pszCurrentDevice[3]);
                        ports.Add(nPort);
                    }
                }

                // Go to next device name
                i += (nLen+1);
            }
        }
        else
        {
            TRACE(_T("Failed in call to QueryDosDevice, GetLastError:%d\n"), 
                  GetLastError());
            return FALSE;
        }
    }
    else
    {
        TRACE(_T("Calling QueryDosDevice on Win9x which does not support \
enumeration of serial ports\n"));
        SetLastError(ERROR_CALL_NOT_IMPLEMENTED);
        return FALSE;
    }

    return TRUE;
}

/*-----------------------------------------------------------------------------
Function:       EnumMethod3

Description:    Uses the GetDefaultCommConfig method to retrieves com port info.

Parameters:     ports - array of available COM port numbers

Return Value:   none
-----------------------------------------------------------------------------*/
BOOL CEnumSer::EnumMethod3(CUIntArray& ports)
{
    // Make sure we clear out any elements which may already be in the array
    ports.RemoveAll();

    // Up to 255 COM ports are supported so we iterate through all of them 
    // seeing if we can get the default configuration
    for (UINT i=1; i<=NUM_PORTS_TO_CHECK; i++)
    {
        // Form the Raw device name
        CString sPort;
        sPort.Format(_T("COM%d"), i);

        COMMCONFIG cc;
        DWORD dwSize = sizeof(COMMCONFIG);
        if (GetDefaultCommConfig(sPort, &cc, &dwSize))
        {
            ports.Add(i);
        }
    }

    // Return the success indicator
    return TRUE;
}

/*-----------------------------------------------------------------------------
Function:       AddData

Description:    Adds an element to the specified array in the ascending order.

Parameters:     arrPorts - array of integers
                uiData - a new value to add to the array

Return Value:   none
-----------------------------------------------------------------------------*/
void CEnumSer::AddData (CUIntArray &arrPorts, UINT uiData)
{
    int i, idx = 0;

    for (i=0; i<arrPorts.GetSize(); i++) 
    {
        if (arrPorts[i] == uiData) 
        {
            idx = -1;
            break;
        }

        if (arrPorts[i] > uiData) 
        {
            idx = i;
            break;
        }
            
        if (i == arrPorts.GetSize() - 1) 
        {
            idx = i+1;
            break;
        }
    }

    if (idx == -1)
    {
        return;
    }

    if (idx < arrPorts.GetSize())
    {
        arrPorts.InsertAt(idx, uiData);
    }
    else
    {
        arrPorts.Add(uiData);
    }
}
